/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System;
using System.Threading;

namespace Inet.Viewer.Data
{
    /// <summary>
    /// 
    /// </summary>
    public enum ProgressMode
    {
        /// <summary>
        /// TotalProgress is unknown, so is just showing that it works
        /// </summary>
        Continuous = 0,

        /// <summary>
        /// TotalProgress is known, so progressing through the steps
        /// </summary>
        Steps = 1
    }

    /// <summary> 
    /// Class for all progresses, which should be handled as a Progress of the report viewer. A progress
    /// encapsulates a special task with a state (such as idle, running or complete) and a value
    /// (23 of 120) of its progress. To start the progress call the <seealso cref="StartProgress()"/> method and
    /// the progress will perform its run method. In this method call <seealso cref="ProgressCount"/>, <seealso cref="TotalProgress"/>,
    /// <seealso cref="Status"/> to propagate the status of the progress.
    /// The progress will be registered at the ProgressPool and will inform its listeners
    /// about property changes such as its status, current step and total steps.
    /// Depending on the progress type the progress will be displayed at the StatusBar
    /// or not. 
    /// </summary>
    public abstract class Progress
    {
        /// <summary>
        /// The type is print Progress, which means the progress will be shown as printing progress in 
        /// the status bar 
        /// </summary>
        public enum ProgressType
        {
            /// <summary>
            /// The type is navigation Progress, which means the progress will be shown as navigation progress in 
            /// the status bar 
            /// </summary>
            PageLoader = 0,

            /// <summary>
            /// The type is print Progress, which means the progress will be shown as printing progress in 
            /// the status bar       
            /// </summary>
            Print = 1,

            /// <summary>
            /// The type is export Progress, which means the progress will be shown as export progress in 
            /// the status bar 
            /// @since 7.0
            /// </summary>
            Export = 2,

            /// <summary>
            /// The type is search Progress, which means the progress will be shown as search progress in 
            /// the status bar 
            /// </summary>
            Search = 3,

            /// <summary>
            /// The type is invisible, do not show this progress in the status bar    
            /// </summary>
            Invisible = 4
        }


        /// <summary>
        /// The different possible status for the Progress
        /// </summary>
        public enum ProgressStatus
        {
            /// <summary>
            /// The progress has not been created yet   
            /// </summary>
            NotInitialized = -1,

            /// <summary>
            /// The progress has been created but has not been started yet  
            /// </summary>
            Initialized = 0,

            /// <summary>
            /// The progress is running   
            /// </summary>
            Running = 1,

            /// <summary>
            /// The progress was successfully completed 
            /// </summary>
            Completed = 2,

            /// <summary>
            /// The progress was canceled   
            /// </summary>
            Canceled = 3,

            /// <summary>
            /// The progress was canceled due to an error   
            /// </summary>
            Error = 4,

            /// <summary>
            /// The progress idles  
            /// </summary>
            Idle = 5
        }

        /// <summary>
        /// Event when the ProgressCount or ProgressTotal changed
        /// </summary>
        public event EventHandler ProgressChanged;

        /// <summary>
        /// Event when the status (ProgressStatus of the Progress changed
        /// </summary>
        public event EventHandler StatusChanged;

        /// <summary>
        /// The property ProgressStatus for PropertyChangeEvents 
        /// </summary>
        public const string PropProgressStatus = "ProgressStatus";

        /// <summary>
        /// The property ProgressTotal for PropertyChangeEvents 
        /// </summary>
        public const string PropProgressTotal = "ProgressTotal";

        /// <summary>
        /// The property ProgressStep for PropertyChangeEvents 
        /// </summary>
        public const string PropProgressStep = "ProgressStep";

        /// <summary>
        /// The property ProgressIndeterminate for PropertyChangeEvents 
        /// </summary>
        public const string PropProgressIndeterminate = "ProgressIndeterminate";

        /// <summary>
        /// the status of the progress </summary>
        private ProgressStatus status = ProgressStatus.NotInitialized;

        /// <summary>
        /// the type of the progress </summary>
        private ProgressType type;
        /// <summary>
        /// the total number of steps to go </summary>
        private int totalProgress;

        /// <summary>
        /// the current step of the progress </summary>
        private int progress;

        /// <summary>
        /// if the progress is an indeterminate progress or not </summary>
        private bool indeterminate = false;

        /// <summary>
        /// the last error message </summary>
        private string errorMessage;

        /// <summary>
        /// the report view to which the progress belongs </summary>
        private Action<Exception> errorDelegate;

        private Thread currentThread;

        /// <summary>
        /// Creates a new progress object with the given type and sets it as being "indeterminate"
        /// (that is, a "busy"/"not busy" progress) or determinate (that is, with clearly
        /// defined steps - step 1 of 5, step 2 of 5, etc.)
        /// The Progress will register itself to the viewer ProgressPool, which handles all progresses. </summary>
        /// <param name="errorDelegate">the error receiver which will be informed about failures during this progress </param>
        /// <param name="type"> E.g. NAVIGATION, PRINT, EXPORT, SEARCH or a custom defined constant.</param>
        public Progress(Action<Exception> errorDelegate, ProgressType type)
            : this(type)
        {
            this.errorDelegate = errorDelegate;
        }

        /// <summary>
        /// Gets the error delegate.
        /// </summary>
        protected Action<Exception> ErrorDelegate
        {
            get
            {
                return errorDelegate;
            }
        }
        
        /// <summary>
        /// Creates a new progress object with the given type and sets it as being "indeterminate"
        /// (that is, a "busy"/"not busy" progress) or determinate (that is, with clearly
        /// defined steps - step 1 of 5, step 2 of 5, etc.)
        /// This Progress will NOT registered to the viewer ProgressPool. It is useful for export without report preview.
        /// </summary>     
        /// <param name="type"> E.g. NAVIGATION, PRINT, EXPORT, SEARCH or a custom defined constant </param>        
        public Progress(ProgressType type)
        {
            this.type = type;
            this.Status = Progress.ProgressStatus.Initialized;
            this.ProgressMode = Data.ProgressMode.Steps;
        }

        /// <summary>
        /// Starts the progress asynchronously in a background thread. </summary>
        /// <exception cref="ViewerException"> when progress is already running</exception>
        public void StartProgress()
        {
            if (currentThread != null && currentThread.IsAlive)
            {
                throw new ViewerException("Progress '" + Name + "' is already running");
            }
            currentThread = new Thread(new ThreadStart(Run));
            Status = ProgressStatus.Running;
            ErrorMessage = null;
            if (indeterminate)
            {
                this.ProgressMode = Data.ProgressMode.Continuous;
            }
            currentThread.Start();
        }

        /// <summary>
        /// Waits until this progress is finished.
        /// </summary>
        public void WaitUntilFinished()
        {
            currentThread.Join();
        }

        /// <summary>
        /// The name of the daemon thread. </summary>
        /// <returns> the name of the daemon thread.</returns>
        public abstract string Name { get; }

        /// <summary>
        /// Call this method to force a cancel of the progress - which must also
        /// set both of the states "finished" and "canceled" to true. 
        /// Please note: some progresses may not be stoppable.
        /// </summary>

        public abstract void Cancel();

        /// <summary>
        /// Returns the type of the progress, which can be one of the following types. </summary>
        /// <returns> Type of the progress </returns>
        /// <seealso cref= "Progress.ProgressType"/>
        public virtual ProgressType Type
        {
            get
            {
                return this.type;
            }
        }

        /// <summary>
        /// Performs this progress.
        /// </summary>
        protected virtual void Run()
        {
        }

        /// <summary>
        /// Returns whether this progress is indeterminate, that is, only a "busy" or "not busy" progress. If not, this progress has
        /// specific steps and can be determined how far it is finished. </summary>
        /// <returns> Whether this progress is indeterminate, that is, only a "busy" or "not busy" progress.</returns>
        public virtual bool Indeterminate
        {
            get
            {
                return this.indeterminate;

            }
            set
            {
                bool old = this.indeterminate;
                this.indeterminate = value;
                if (old != value)
                {
                    this.OnStatusChanged();
                }
            }
        }


        /// <summary>
        /// Call this method to set the status of this progress. 
        /// The status could be <seealso cref="ProgressStatus"/>  
        /// All registered listeners will be informed about this change. </summary>
        /// <param> Status of this progress </param> 
        public virtual ProgressStatus Status
        {
            set
            {
                ProgressStatus oldValue = this.status;
                this.status = value;
                if (oldValue != value)
                {
                    OnStatusChanged();
                }
            }
            get
            {
                return this.status;
            }
        }

        /// <summary>
        /// Return true if the progress has stopped its task. The status of the Progress can be complete, canceled or error. </summary>
        /// <returns> whether the progress has stopped</returns>
        public virtual bool Finished
        {
            get
            {
                bool finished = this.status == Progress.ProgressStatus.Canceled || this.status == ProgressStatus.Completed || this.status == ProgressStatus.Error;
                return finished;
            }
        }


        /// <summary>
        /// Use this method to inform all registered listeners about the total steps (e.g. pages)
        /// to go. If total is set to zero the progress will be declared as inactive (status = Progress.STATUS_IDLE). </summary>
        /// <param> steps to go</param>
        public virtual int TotalProgress
        {
            set
            {
                int old = this.totalProgress;
                this.totalProgress = value;
                if (value == 0)
                {
                    this.Status = ProgressStatus.Idle;
                }
                else if (this.Status == ProgressStatus.Idle)
                {
                    Status = ProgressStatus.Running;
                }

                if (old != value)
                {
                    OnProgressChanged();
                }
            }
            get
            {
                return this.totalProgress;
            }
        }

        /// <summary>
        /// returns the ProgressMode, by default it is the step wise progress.
        /// </summary>
        public virtual ProgressMode ProgressMode
        {
            get;
            protected set;
        }


        /// <summary>
        /// Use this method to set the progress and
        /// to inform all registered listeners about the current step. </summary>
        /// <param> Steps made in the progress so far (1-based) </param>
        public virtual int ProgressCount
        {
            set
            {
                int oldStep = this.progress;
                this.progress = value;
                if (oldStep != value)
                {
                    OnProgressChanged();
                }
            }
            get
            {
                return this.progress;
            }
        }

        /// <summary>
        /// Returns the last error message of this Progress. </summary>
        /// <returns> The last error of this Progress.</returns>
        public virtual string ErrorMessage
        {
            get
            {
                return this.errorMessage;
            }
            set
            {
                this.errorMessage = value;
            }
        }


        /// <summary>
        /// Shows an error dialog for this error message. If there is no ReportView it will print a stacktrace to the console.
        /// It is calling SetErrorMessage(String) and SetStatus(STATUS_ERROR) before the error message is show. </summary>
        /// <param name="th"> The exception to show in the error dialog. </param>
        /// <seealso cref= "Status"/>  
        public virtual void ShowError(Exception th)
        {
            try
            {
                ErrorMessage = th.Message;
                Status = ProgressStatus.Error;
                if (errorDelegate != null)
                {
                    errorDelegate(th);
                }
                ViewerUtils.PrintStackTrace(th);
            }
            catch (Exception ex)
            {
                // ignore exceptions during error handling
                ViewerUtils.PrintStackTrace(ex);
            }
        }


        /// <summary>
        /// 
        /// </summary>
        public void OnStatusChanged()
        {
            if (this.StatusChanged != null)
            {
                StatusChanged(this, new EventArgs());
            }
        }

        /// <summary>
        /// 
        /// </summary>
        public void OnProgressChanged()
        {
            if (this.ProgressChanged != null)
            {
                ProgressChanged(this, new EventArgs());
            }
        }
    }
}